﻿
/****************************************************************************/
/*Copyright (c) 2011, Florent DEVILLE.                                      */
/*All rights reserved.                                                      */
/*                                                                          */
/*Redistribution and use in source and binary forms, with or without        */
/*modification, are permitted provided that the following conditions        */
/*are met:                                                                  */
/*                                                                          */
/* - Redistributions of source code must retain the above copyright         */
/*notice, this list of conditions and the following disclaimer.             */
/* - Redistributions in binary form must reproduce the above                */
/*copyright notice, this list of conditions and the following               */
/*disclaimer in the documentation and/or other materials provided           */
/*with the distribution.                                                    */
/* - The names of its contributors cannot be used to endorse or promote     */
/*products derived from this software without specific prior written        */
/*permission.                                                               */
/* - The source code cannot be used for commercial purposes without         */
/*its contributors' permission.                                             */
/*                                                                          */
/*THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT         */
/*LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS         */
/*FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE            */
/*COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,       */
/*INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,      */
/*BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;          */
/*LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER          */
/*CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT        */
/*LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN         */
/*ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/*POSSIBILITY OF SUCH DAMAGE.                                               */
/****************************************************************************/

using GE.Visualisation;
using GE.Physics.Shapes;
using GE.Physics;
using GE.Manager;
using Microsoft.Xna.Framework;
//using Microsoft.Xna.Framework.Graphics;

namespace GE.World.Entities
{
    class SliderEntity : EnemyEntity
    {
        enum eState
        {
            eStateMove,
            eStateIdle,
            eStateCount
        }

        /// <summary>
        /// Entity inner state
        /// </summary>
        eState _innerState;

        /// <summary>
        /// Animation's id
        /// </summary>
        int _iIdAnimation;

        /// <summary>
        /// Left edge
        /// </summary>
        Vector2 _v2EdgeLeft;

        /// <summary>
        /// Right edge
        /// </summary>
        Vector2 _v2EdgeRight;

        /// <summary>
        /// Physic shape
        /// </summary>
        //DynamicShapeRectangle _shape;

        /// <summary>
        /// Entity speed
        /// </summary>
        int _iSpeed;

        /// <summary>
        /// Direction the entity is facing
        /// </summary>
        int _iDirection;

        /// <summary>
        /// Life time of the idle state
        /// </summary>
        int _iIdleStateLifeTime;

        /// <summary>
        /// Birth time of the idle state
        /// </summary>
        int _iIdleStateBirthTime;

        /// <summary>
        /// Flag set when the state was changed
        /// </summary>
        bool _bPreState;

        /// <summary>
        /// Animation current frame
        /// </summary>
        int _iAnimationCurrentFrame;

        /// <summary>
        /// Animation current time
        /// </summary>
        int _iAnimationCurrentTime;

        /// <summary>
        /// Id of the explosion animation
        /// </summary>
        int _iIdAnimationExplosion;

        /// <summary>
        /// The position offset to display the explosion
        /// </summary>
        Vector2 _v2AnimationExplosionOffsetPosition;

        #region Properties
#if !GAME
        public static string EDITOR_TILESET { get { return "enemysheet.xml"; } }
        public static string EDITOR_SPRITE { get { return "slider_3"; } }
#endif
        public Vector2 EdgeLeft { set { _v2EdgeLeft = value; } }
        public Vector2 EdgeRight { set { _v2EdgeRight = value; } }
        public int Speed { set { _iSpeed = value; } }
        public int IdleLiftTime { set { _iIdleStateLifeTime = value; } }

        #endregion

        /// <summary>
        /// Constructor
        /// </summary>
        public SliderEntity()
            : base()
        {
            _iIdAnimation = -1;
            _v2EdgeLeft = Vector2.Zero;
            _v2EdgeRight = Vector2.Zero;
            _iSpeed = 0;
            _shape = Physics.Physics.Instance.createDynamicRectangle(0, 0, Vector2.Zero, this);
            _shape._bCollisionEnable = false;
            _iDirection = -1;
            _iIdleStateBirthTime = -1;
            _iIdleStateLifeTime = -1;
            _iAnimationCurrentFrame = -1;
            _iAnimationCurrentTime = -1;
            _iIdAnimationExplosion = -1;
            _v2AnimationExplosionOffsetPosition = Vector2.Zero;
        }

        /// <summary>
        /// Initialise the entity. Load all the data which are not loaded during the level loading.
        /// </summary>
        public override void init()
        {
            _iIdAnimation = Visu.Instance.getAnimationID("Slider");
            _iIdAnimationExplosion = Visu.Instance.getAnimationID("Little_Explosion");

            int iIdTexture = Visu.Instance.getAnimation(_iIdAnimation).indexTexture;
            int iIdSprite = Visu.Instance.getAnimation(_iIdAnimation).idFirstSprite;

            int iWidth = Visu.Instance.getSpriteWidth(iIdTexture, iIdSprite);
            int iHeight = Visu.Instance.getSpriteHeight(iIdTexture, iIdSprite);
            _shape.resize(iWidth, iHeight);
            _shape._iGroup = (int)ePhysicGroup.ePhysicEnemy;

            _v2AnimationExplosionOffsetPosition = new Vector2(iWidth / 2, iHeight / 2);

            base.init();
        }
        /// <summary>
        /// Activator
        /// </summary>
        public override void activate()
        {
            _shape._bCollisionEnable = true;
            _bActive = true;
            _iDirection = 1;
            _v2Position = _v2EdgeLeft;
            _shape._v2position = _v2Position;
            
            _innerState = eState.eStateIdle;
            _bPreState = true;

        }

        /// <summary>
        /// Update
        /// </summary>
        public override void update()
        {
            switch (_innerState)
            {
                case eState.eStateIdle:
                    updateStateIdle();
                    break;
                case eState.eStateMove:
                    updateStateMove();
                    break;
            }

            //check collision
            CollisionResult res = Physics.Physics.Instance.checkFirstRegisteredCollisionEx(_shape, (int)ePhysicGroup.ePhysicPlayer);
            if (res != null) res.Entity.hurt(_iDamages);
            
        }

        /// <summary>
        /// Update the idle state
        /// </summary>
        public void updateStateIdle()
        {
            if (_bPreState)
            {
                _bPreState = false;
                _iIdleStateBirthTime = TimeClock.Clock.instance.millisecs;
            }

            if (TimeClock.Clock.instance.millisecs >= _iIdleStateBirthTime + _iIdleStateLifeTime)
            {
                _bPreState = true;
                _innerState = eState.eStateMove;
            }
        }

        /// <summary>
        /// Update the move state
        /// </summary>
        public void updateStateMove()
        {
            if (_bPreState)
            {
                _bPreState = false;
                _iDirection *= -1;
            }


            if (_iDirection > 0)
            {
                Vector2 direction = _v2EdgeRight - _v2EdgeLeft;
                direction.Normalize();

                float fDistance = (_v2Position - _v2EdgeRight).Length();
                if (fDistance < _iSpeed)
                {
                    _v2Position = _v2EdgeRight;
                    _shape._v2position = _v2Position;
                    _innerState = eState.eStateIdle;
                    _bPreState = true;
                    return;
                }
                else
                {
                    _v2Position += direction * _iSpeed;
                    _shape._v2position = _v2Position;
                }
            }
            else
            {
                Vector2 direction = _v2EdgeLeft - _v2EdgeRight;
                direction.Normalize();

                float fDistance = (_v2Position - _v2EdgeLeft).Length();
                if (fDistance < _iSpeed)
                {
                    _v2Position = _v2EdgeLeft;
                    _shape._v2position = _v2Position;
                    _innerState = eState.eStateIdle;
                    _bPreState = true;
                    return;
                }
                else
                {
                    _v2Position += direction * _iSpeed;
                    _shape._v2position = _v2Position;
                }
            }
        }

        /// <summary>
        /// Render
        /// </summary>
        public override void render()
        {
            Visu.Instance.displayAnimation(_iIdAnimation, ScreenPosition, ref _iAnimationCurrentFrame, ref _iAnimationCurrentTime);
        }

        public override void die()
        {
            _shape._bCollisionEnable = false;
            _bActive = false;
            Manager.ExplosionManager.Instance.activate(_iIdAnimationExplosion, Position + _v2AnimationExplosionOffsetPosition);
        }

        //public override void hurt(int damages)
        //{
        //    HP -= damages;
        //    if(HP <= 0) die();
        //}
    }
}
